from codeable_detectors.evidences import Evidence


class DetectorResults(object):
    def __init__(self, results=None, failed_evidences=None):
        self.evidences = []
        if results:
            self.add_evidences(results)
        self.failed_evidences = []
        if failed_evidences:
            self.add_failed(failed_evidences)

    def __str__(self):
        return ("DetectorResults: Evidences = [" +
                ", ".join(str(evidence) for evidence in self.evidences) +
                "], failed_evidences = [" +
                ", ".join(str(evidence) for evidence in self.failed_evidences) +
                "]")

    def add_evidences(self, results_to_add):
        if isinstance(results_to_add, DetectorResults):
            if results_to_add.failed_evidences:
                raise Exception("cannot add detector results that have failed, use add_failed instead")
            self.evidences.extend(results_to_add.evidences)
        elif isinstance(results_to_add, Evidence):
            self.evidences.append(results_to_add)
        elif isinstance(results_to_add, list) and all(isinstance(r, Evidence) for r in results_to_add):
            self.evidences.extend(results_to_add)
        else:
            raise Exception("unknown types provided for add, require Evidence types but got: '" +
                            str(results_to_add) + "'")

    def add_failed(self, failed, prepend_to_failed_string=None):
        if prepend_to_failed_string:
            failed.prepend_to_failed(prepend_to_failed_string)
        if isinstance(failed, DetectorResults):
            self.failed_evidences.extend(failed.failed_evidences)
        elif isinstance(failed, str):
            self.failed_evidences.append(failed)
        elif isinstance(failed, list) and all(isinstance(f, str) for f in failed):
            self.failed_evidences.extend(failed)
        else:
            raise Exception("unknown types provided for add_failed, require string types but got: '" +
                            str(failed) + "'")

    # add a detector results and decide based on it whether to add it as failed evidences
    # or evidences
    def add_detector_results(self, results_to_add, prepend_to_failed_string=None):
        if results_to_add.have_failed():
            self.add_failed(results_to_add, prepend_to_failed_string)
        else:
            self.add_evidences(results_to_add)

    def no_evidence_found(self):
        if not self.evidences:
            return True
        return False

    def have_failed(self):
        if self.failed_evidences:
            return True
        return False

    def succeeded(self):
        if not self.failed_evidences and self.evidences:
            return True
        return False

    def get_all_matches(self):
        matches = []
        for evidence in self.evidences:
            matches.extend(evidence.matches)
        return matches

    def prepend_to_failed(self, prepend_string):
        new_failed = []
        for failed_string in self.failed_evidences:
            new_failed.append(prepend_string + failed_string)
        self.failed_evidences = new_failed


def detector_failed(msg):
    return DetectorResults(failed_evidences=msg)
